#include "Triangle.h"

Triangle::Triangle()
{
	// triangle
	left[0] = -0.5;
	left[1] = 0.0;
	right[0] = 0.5;
	right[1] = 0.0;
	top[0] = 0.0;
	top[1] = 0.5;

	convexHull = new float*[8];
	for (int x = 0; x < 8; x++)
	{
		convexHull[x] = new float[2];
	}

	prev = 0;
	next = 0;

	CalcHull();
}

Triangle::Triangle(Triangle* triangle)
{
	// triangle
	left[0] = triangle->left[0];
	left[1] = triangle->left[1];
	right[0] = triangle->right[0];
	right[1] = triangle->right[1];
	top[0] = triangle->top[0];
	top[1] = triangle->top[1];

	convexHull = new float*[8];
	for (int x = 0; x < 8; x++)
	{
		convexHull[x] = new float[2];
		convexHull[x][0] = triangle->convexHull[x][0];
		convexHull[x][1] = triangle->convexHull[x][1];
	}

	prev = triangle->prev;
	next = triangle->next;
}

Triangle::Triangle( float* _left, float* _right, float* _top ) 
{
	left[0] = _left[0];
	left[1] = _left[1];
	right[0] = _right[0];
	right[1] = _right[1];
	top[0] = _top[0];
	top[1] = _top[1];

	convexHull = new float*[8];
	for (int x = 0; x < 8; x++)
	{
		convexHull[x] = new float[2];
	}
	
	CalcHull();

	prev = 0;
	next = 0;
}

Triangle::~Triangle()
{
	for (int x = 0; x < 8; x++)
	{
		delete[] convexHull[x];
	}

	delete[] convexHull;
}

float Triangle::BaseLength()
{
	float xTerm = left[0] - right[0];
	float yTerm = left[1] - right[1];
	return (float) sqrt( xTerm * xTerm + yTerm * yTerm );
}

float Triangle::SideLength()
{
	float xTerm = left[0] - top[0];
	float yTerm = left[1] - top[1];
	return (float) sqrt( xTerm * xTerm + yTerm * yTerm );
}

void Triangle::NextLevel( Triangle* rightTriangle )
{
	rightTriangle->left[0] = top[0];
	rightTriangle->left[1] = top[1];
	rightTriangle->right[0] = right[0];
	rightTriangle->right[1] = right[1];
	rightTriangle->CalcTop();
	rightTriangle->CalcHull();

	right[0] = top[0];
	right[1] = top[1];
	CalcTop();
	CalcHull();

	next = rightTriangle;
	rightTriangle->prev = this;
}

void Triangle::CalcTop()
{
	// calculate midpoint of base
	float midpoint[2];
	midpoint[0] = (float)((left[0] + right[0]) / 2.0);
	midpoint[1] = (float)((left[1] + right[1]) / 2.0);
	
	// translate left datapoint to origin
	float translatedLeft[2];
	translatedLeft[0] = left[0] - midpoint[0];
	translatedLeft[1] = left[1] - midpoint[1];

	top[0] = translatedLeft[1];
	top[1] = translatedLeft[0];

	top[1] = -top[1];

	// translate back to original position
	top[0] = top[0] + midpoint[0];
	top[1] = top[1] + midpoint[1];
}

void Triangle::CalcHull()
{

	// datapoint order:
	//				0					1
	//				-------------------
	//			  /						\
	//			/						  \
	//		  /								\
	//	 7	/								  \  2
	//	   |								   |
	//	   |								   |
	//	 6 |								   | 3
	//		\								  /
	//		5 \	____________________________/ 4

	// calculate midpoint of base
	float midpoint[2];
	midpoint[0] = (float)((left[0] + right[0]) / 2.0);
	midpoint[1] = (float)((left[1] + right[1]) / 2.0);

	// translate datapoints to origin
	float translatedLeft[2];
	translatedLeft[0] = left[0] - midpoint[0];
	translatedLeft[1] = left[1] - midpoint[1];

	float translatedTop[2];
	translatedTop[0] = top[0] - midpoint[0];
	translatedTop[1] = top[1] - midpoint[1];

	// calculate convex hull position
	convexHull[0][0] = 2 * translatedTop[0] + translatedLeft[0];
	convexHull[0][1] = 2 * translatedTop[1] + translatedLeft[1];

	convexHull[1][0] = 2 * translatedTop[0] - translatedLeft[0];
	convexHull[1][1] = 2 * translatedTop[1] - translatedLeft[1];

	convexHull[2][0] = translatedTop[0] - 2 * translatedLeft[0];
	convexHull[2][1] = translatedTop[1] - 2 * translatedLeft[1];

	convexHull[3][0] = -2 * translatedLeft[0];
	convexHull[3][1] = -2 * translatedLeft[1];

	convexHull[4][0] = (float) (-0.5 * translatedTop[0] - 1.5 * translatedLeft[0]);
	convexHull[4][1] = (float) (-0.5 * translatedTop[1] - 1.5 * translatedLeft[1]);

	convexHull[5][0] = (float) (-0.5 * translatedTop[0] + 1.5 * translatedLeft[0]);
	convexHull[5][1] = (float) (-0.5 * translatedTop[1] + 1.5 * translatedLeft[1]);

	convexHull[6][0] = 2 * translatedLeft[0];
	convexHull[6][1] = 2 * translatedLeft[1];

	convexHull[7][0] = translatedTop[0] + 2 * translatedLeft[0];
	convexHull[7][1] = translatedTop[1] + 2 * translatedLeft[1];

	// translate hull back to midpoint
	convexHull[0][0] += midpoint[0];	convexHull[0][1] += midpoint[1];
	convexHull[1][0] += midpoint[0];	convexHull[1][1] += midpoint[1];
	convexHull[2][0] += midpoint[0];	convexHull[2][1] += midpoint[1];
	convexHull[3][0] += midpoint[0];	convexHull[3][1] += midpoint[1];
	convexHull[4][0] += midpoint[0];	convexHull[4][1] += midpoint[1];
	convexHull[5][0] += midpoint[0];	convexHull[5][1] += midpoint[1];
	convexHull[6][0] += midpoint[0];	convexHull[6][1] += midpoint[1];
	convexHull[7][0] += midpoint[0];	convexHull[7][1] += midpoint[1];	
}

void Triangle::Set(Triangle* set)
{
	left[0] = set->left[0];
	left[1] = set->left[1];
	right[0] = set->right[0];
	right[2] = set->right[1];
	top[0] = set->top[0];
	top[1] = set->top[1];
	CalcHull();
}