#include "TriangleList.h"

TriangleList::TriangleList()
{
	Init();
}

TriangleList::TriangleList(double left0,
						   double left1,
						   double right0,
						   double right1)
{
	Init();
	add(left0, left1, right0, right1);
}

void TriangleList::Init()
{
	triangles = 50000;
	size = 0;
	allocated = triangles * 6;

	triangleList = (double*)malloc(sizeof(double) * allocated);
	hullList = (double*)malloc(sizeof(double) * triangles * 16);
	triangleIndices = (unsigned int*)malloc(sizeof(unsigned int) * allocated);
	frustum = (double *)malloc(sizeof(double) * 8);

	for (int x = 0; x < allocated; x++)
	{
		triangleIndices[x] = x;
	}

	vertexHash.Clear();
	centerHash.Clear();
}

TriangleList::~TriangleList()
{
	free(triangleList);
	free(hullList);
	free(triangleIndices);
	free(frustum);
}

void TriangleList::Dump(ostream& os)
{
	for (int x = 0; x < size; x++)
	{
		os.write((char *) &triangleList[6 * x], sizeof(double));
		os.write((char *) &triangleList[6 * x + 1], sizeof(double));
		os.write((char *) &triangleList[6 * x + 2], sizeof(double));
		os.write((char *) &triangleList[6 * x + 3], sizeof(double));
	}
}

void TriangleList::Render()
{
	glVertexPointer(2, GL_DOUBLE, 0, triangleList);
	glDrawElements(GL_TRIANGLES, size * 3, GL_UNSIGNED_INT, triangleIndices);
}

int TriangleList::GetSize()
{
	return size;
}

void TriangleList::Reset()
{
	size = 0;
	vertexHash.Clear();
	centerHash.Clear();
}

bool TriangleList::Split()
{
	vertexHash.Clear();
	centerHash.Clear();

	if (size == 0) return true;
	if (2 * size >= triangles) return false;

	int end = size;
	for (int x = 0; x < end; x++)
	{
		NextLevel(&triangleList[6 * x], &triangleList[6 * size], 
				  &hullList[16 * x], &hullList[16 * size]);
		size++;
	}
	return true;
}

void TriangleList::add(double left0, double left1, double right0, double right1)
{
	int index = 6 * size;
	triangleList[index] = left0;
	triangleList[index + 1] = left1;
	triangleList[index + 2] = right0;
	triangleList[index + 3] = right1;
	CalcTop(&triangleList[index]);
	CalcHull(&triangleList[index], &hullList[16 * size]);

	vertexHash.Insert(triangleList[index], triangleList[index + 1]);
	vertexHash.Insert(triangleList[index + 2], triangleList[index + 3]);
	vertexHash.Insert(triangleList[index + 4], triangleList[index + 5]);

	// calculate center
	double mid1x = (triangleList[index] + triangleList[index + 2]) * 0.5;
	double mid1y = (triangleList[index + 1] + triangleList[index + 3]) * 0.5;
	double mid2x = (triangleList[index + 4] + mid1x) * 0.5;
	double mid2y = (triangleList[index + 5] + mid1y) * 0.5;
	centerHash.Insert(mid2x, mid2y);

	size++;
}

void TriangleList::add(double* triangle, double* hull)
{
	StompTriangle(&triangleList[6 * size], triangle);
	StompHull(&hullList[16 * size], hull);
	size++;
}

void TriangleList::CompactVisible(double top, double bottom, double left, double right)
{
	int tail = 0;
	double* test;
	vertexHash.Clear();
	centerHash.Clear();

	for (int scan = 0; scan<size; scan++)
	{
		test = &hullList[16*scan];
		if (IsVisible(test, top, bottom, left, right))
		{
			StompTriangle(&triangleList[6*tail], &triangleList[6*scan]);
			StompHull(&hullList[16*tail], test);
			tail++;
		}
	}
	size = tail;
}

// see if visibleHull is visible from (top, bottom, left right) frustum
bool TriangleList::IsVisible( double* convexHull, double top,
							  double bottom, double left, double right)
{
	// view frustum
	//
	//		0				  1
	//	(left, top)		(right, top)
	//
	//		---------------------
	//		|					|
	//		|					|
	//		|					|
	//		|					|
	//		|					|
	//		|					|
	//		|					|
	//		---------------------
	//
	//		2				  3
	//	(left, bottom)  (right, bottom)


	// calculate view frustum
	frustum[0] = left;
	frustum[1] = top;
	frustum[2] = right;
	frustum[3] = top;
	frustum[4] = right;
	frustum[5] = bottom;
	frustum[6] = left;
	frustum[7] = bottom;

	return TriangleList::Intersection( frustum, 4, convexHull, 8 );
}

// calculates if polygonPoints1 intersects polygonPoints2
bool TriangleList::Intersection(double* polygonPoints1, int totalPoints1,
								double* polygonPoints2, int totalPoints2 )
{
	int x;
	bool inside = false;

	// check if any polygon2 points are inside polygon1
	for (x = 0; x < totalPoints2; x++)
	{
		inside = TriangleList::IsInside( &polygonPoints2[2 * x],
										 polygonPoints1,
										 totalPoints1 );
		if ( inside )
		{
			return true;
		}
	}


	// check if any polygon1 points are inside polygon2
	for (x = 0; x < totalPoints1; x++)
	{
		inside = TriangleList::IsInside( &polygonPoints1[2 * x],
										 polygonPoints2,
										 totalPoints2 );
		if ( inside )
		{
			return true;
		}
	}

	return false;
}

// calculates if insideTest is inside polygonPoints
bool TriangleList::IsInside( double* insideTest,
							 double* polygonPoints,
							 int totalPoints )
{
	int x;
	
	// copy the polygon points
	double* _polygonPoints;
	_polygonPoints = new double[totalPoints * 2];
	for (x = 0; x < totalPoints; x++)
	{
		_polygonPoints[2 * x] = polygonPoints[2 * x]; 
		_polygonPoints[2 * x + 1] = polygonPoints[2 * x + 1]; 
	}

	// transform points to origin
	for (x = 0; x < totalPoints; x++)
	{
		_polygonPoints[2 * x] -= insideTest[0];
		_polygonPoints[2 * x + 1] -= insideTest[1];
	}

	// sum polygon angles
	double thetaTotal = 0;
	for (x = 0; x < totalPoints; x++)
	{
		// calc angle
		double vector1x = _polygonPoints[2 * x];
		double vector1y = _polygonPoints[2 * x + 1];
		double vector2x = _polygonPoints[2 * ((x + 1) % totalPoints)];
		double vector2y = _polygonPoints[2 * ((x + 1) % totalPoints) + 1];
		double theta = (double) acos(((vector1x * vector2x) + (vector1y * vector2y)) /
									 (sqrt((vector1x * vector1x) + (vector1y * vector1y)) *
									  sqrt((vector2x * vector2x) + (vector2y * vector2y))) );
	
		// calc angle direction
		double sign = vector1x * vector2y - vector1y * vector2x;
		theta = (double) fabs( theta );
		if ( sign < 0 )
		{
			theta = -theta;
		}
		thetaTotal += theta;
	}
	thetaTotal = (double) fabs( thetaTotal );
	delete[] _polygonPoints;

	// check if point is outside
	if ( thetaTotal < EPSILON )
	{
		return false;
	}

	return true;
}

void TriangleList::CalcTop(double* toCalc)
{
	// calculate midpoint of base
	double midpoint[2];
	midpoint[0] = (double)((toCalc[0] + toCalc[2]) * 0.5);
	midpoint[1] = (double)((toCalc[1] + toCalc[3]) * 0.5);
	
	// translate left datapoint to origin
	double translatedLeft[2];
	translatedLeft[0] = toCalc[0] - midpoint[0];
	translatedLeft[1] = toCalc[1] - midpoint[1];

	toCalc[4] = translatedLeft[1];
	toCalc[5] = translatedLeft[0];

	toCalc[5] = -toCalc[5];

	// translate back to original position
	toCalc[4] = toCalc[4] + midpoint[0];
	toCalc[5] = toCalc[5] + midpoint[1];
}

void TriangleList::CalcHull(double* triangle, double* convexHull)
{

	// datapoint order:
	//				0					1
	//				-------------------
	//			  /						\
	//			/						  \
	//		  /								\
	//	 7	/								  \  2
	//	   |								   |
	//	   |								   |
	//	 6 |								   | 3
	//		\								  /
	//		5 \	____________________________/ 4

	// calculate midpoint of base
	double midpoint[2];
	midpoint[0] = (double)((triangle[0] + triangle[2]) * 0.5);
	midpoint[1] = (double)((triangle[1] + triangle[3]) * 0.5);

	// translate datapoints to origin
	double translatedLeft[2];
	translatedLeft[0] = triangle[0] - midpoint[0];
	translatedLeft[1] = triangle[1] - midpoint[1];

	double translatedTop[2];
	translatedTop[0] = triangle[4] - midpoint[0];
	translatedTop[1] = triangle[5] - midpoint[1];

	// calculate convex hull position
	convexHull[0] = 2 * translatedTop[0] + translatedLeft[0];
	convexHull[1] = 2 * translatedTop[1] + translatedLeft[1];

	convexHull[2] = 2 * translatedTop[0] - translatedLeft[0];
	convexHull[3] = 2 * translatedTop[1] - translatedLeft[1];

	convexHull[4] = translatedTop[0] - 2 * translatedLeft[0];
	convexHull[5] = translatedTop[1] - 2 * translatedLeft[1];

	convexHull[6] = -2 * translatedLeft[0];
	convexHull[7] = -2 * translatedLeft[1];

	convexHull[8] = (float) (-0.5 * translatedTop[0] - 1.5 * translatedLeft[0]);
	convexHull[9] = (float) (-0.5 * translatedTop[1] - 1.5 * translatedLeft[1]);

	convexHull[10] = (float) (-0.5 * translatedTop[0] + 1.5 * translatedLeft[0]);
	convexHull[11] = (float) (-0.5 * translatedTop[1] + 1.5 * translatedLeft[1]);

	convexHull[12] = 2 * translatedLeft[0];
	convexHull[13] = 2 * translatedLeft[1];

	convexHull[14] = translatedTop[0] + 2 * translatedLeft[0];
	convexHull[15] = translatedTop[1] + 2 * translatedLeft[1];

	// translate hull back to midpoint
	convexHull[0] += midpoint[0];	convexHull[1] += midpoint[1];
	convexHull[2] += midpoint[0];	convexHull[3] += midpoint[1];
	convexHull[4] += midpoint[0];	convexHull[5] += midpoint[1];
	convexHull[6] += midpoint[0];	convexHull[7] += midpoint[1];
	convexHull[8] += midpoint[0];	convexHull[9] += midpoint[1];
	convexHull[10] += midpoint[0];	convexHull[11] += midpoint[1];
	convexHull[12] += midpoint[0];	convexHull[13] += midpoint[1];
	convexHull[14] += midpoint[0];	convexHull[15] += midpoint[1];	
}

void TriangleList::StompTriangle(double* stompee, double* stomper)
{
	if (stompee == stomper) return;

	stompee[0] = stomper[0];
	stompee[1] = stomper[1];
	stompee[2] = stomper[2];
	stompee[3] = stomper[3];
	stompee[4] = stomper[4];
	stompee[5] = stomper[5];

	vertexHash.Insert(stomper[0], stomper[1]);
	vertexHash.Insert(stomper[2], stomper[3]);
	vertexHash.Insert(stomper[4], stomper[5]);

	// calculate center
	double mid1x = (stomper[0] + stomper[2]) * 0.5;
	double mid1y = (stomper[1] + stomper[3]) * 0.5;
	double mid2x = (stomper[4] + mid1x) * 0.5;
	double mid2y = (stomper[5] + mid1y) * 0.5;
	centerHash.Insert(mid2x, mid2y);
}

void TriangleList::StompHull(double* stompee, double* stomper)
{
	if (stompee == stomper) return;

	stompee[0] = stomper[0];	stompee[1] = stomper[1];
	stompee[2] = stomper[2];	stompee[3] = stomper[3];
	stompee[4] = stomper[4];	stompee[5] = stomper[5];
	stompee[6] = stomper[6];	stompee[7] = stomper[7];
	stompee[8] = stomper[8];	stompee[9] = stomper[9];
	stompee[10] = stomper[10];	stompee[11] = stomper[11];
	stompee[12] = stomper[12];	stompee[13] = stomper[13];
	stompee[14] = stomper[14];	stompee[15] = stomper[15];
}

void TriangleList::NextLevel(double* triangleSplit, double* triangleStore,
							 double* hullSplit, double* hullStore)
{
	triangleStore[0] = triangleSplit[4];
	triangleStore[1] = triangleSplit[5];
	triangleStore[2] = triangleSplit[2];
	triangleStore[3] = triangleSplit[3];
	CalcTop(triangleStore);
	CalcHull(triangleStore, hullStore);
	vertexHash.Insert(triangleStore[0], triangleStore[1]);
	vertexHash.Insert(triangleStore[2], triangleStore[3]);
	vertexHash.Insert(triangleStore[4], triangleStore[5]);

	// calculate center
	double mid1x = (triangleStore[0] + triangleStore[2]) * 0.5;
	double mid1y = (triangleStore[1] + triangleStore[3]) * 0.5;
	double mid2x = (triangleStore[4] + mid1x) * 0.5;
	double mid2y = (triangleStore[5] + mid1y) * 0.5;
	centerHash.Insert(mid2x, mid2y);

	triangleSplit[2] = triangleSplit[4];
	triangleSplit[3] = triangleSplit[5];
	CalcTop(triangleSplit);
	CalcHull(triangleSplit, hullSplit);
	vertexHash.Insert(triangleSplit[0], triangleSplit[1]);
	vertexHash.Insert(triangleSplit[2], triangleSplit[3]);
	vertexHash.Insert(triangleSplit[4], triangleSplit[5]);

	// calculate center
	mid1x = (triangleSplit[0] + triangleSplit[2]) * 0.5;
	mid1y = (triangleSplit[1] + triangleSplit[3]) * 0.5;
	mid2x = (triangleSplit[4] + mid1x) * 0.5;
	mid2y = (triangleSplit[5] + mid1y) * 0.5;
	centerHash.Insert(mid2x, mid2y);
}

void TriangleList::BuildInterior(TriangleList* interior)
{
	interior->Reset();
	for (int x = 0; x < size; x++)
	{
		if (IsInterior(&triangleList[6 * x]))
		{
			interior->add(&triangleList[6 * x], &hullList[16 * x]);
		}
	}
}

void TriangleList::BuildBarriers(TriangleList* interior, BarrierList* bList)
{
	bList->Clear();
	int insideSize = interior->GetSize();
	double* insideList = interior->triangleList;

	for (int x = 0; x < insideSize; x++)
	{
		CalcBarrier(&insideList[6 * x], bList);
	}
}

// Check if triangle is in interior
bool TriangleList::IsInterior(double* currentTriangle)
{
	int topHits = vertexHash.IsMember( currentTriangle[4],
									   currentTriangle[5] );
	int leftHits = vertexHash.IsMember( currentTriangle[0],
									    currentTriangle[1] );
	int rightHits = vertexHash.IsMember( currentTriangle[2],
										 currentTriangle[3] );

	if( ( topHits + leftHits + rightHits ) == 20 )
	{
		return true;
	}

	return false;
}

void TriangleList::CalcBarrier(double* currentTriangle, BarrierList* bList)
{
	float baseMid[2];
	float upVector[2];
	float leftVector[2];
	float rightVector[2];
	float c1[2];
	float c2[2];
	float c3[2];

	baseMid[0] = (currentTriangle[0] + currentTriangle[2]) * 0.5;
	baseMid[1] = (currentTriangle[1] + currentTriangle[3]) * 0.5;
	upVector[0] = currentTriangle[4] - baseMid[0];
	upVector[1] = currentTriangle[5] - baseMid[1];
	leftVector[0] = currentTriangle[0] - baseMid[0];
	leftVector[1] = currentTriangle[1] - baseMid[1];
	rightVector[0] = currentTriangle[2] - baseMid[0];
	rightVector[1] = currentTriangle[3] - baseMid[1];

	c1[0] = baseMid[0] - 1.5 * upVector[0];
	c1[1] = baseMid[1] - 1.5 * upVector[1];
	if (!centerHash.IsMember(c1[0], c1[1]))
	{
		bList->add(&currentTriangle[0], &currentTriangle[2]);
	}

	c2[0] = baseMid[0] + 2 * leftVector[0] + 1.5 * upVector[0];
	c2[1] = baseMid[1] + 2 * leftVector[1] + 1.5 * upVector[1];
	if (!centerHash.IsMember(c2[0], c2[1]))
	{
		bList->add(&currentTriangle[0], &currentTriangle[4]);
	}

	c3[0] = baseMid[0] + 2 * rightVector[0] + 1.5 * upVector[0];
	c3[1] = baseMid[1] + 2 * rightVector[1] + 1.5 * upVector[1];
	if (!centerHash.IsMember(c3[0], c3[1]))
	{
		bList->add(&currentTriangle[4], &currentTriangle[2]);
	}

	return;
}