#include <windows.h>
#include <gl/gl.h>
#include <math.h>
#include <fstream.h>
#include "resource.h"
#include "Dragon.h"
#include "Camera.h"

// Function Declarations
LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );
BOOL CALLBACK ToolbarDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam );
BOOL OpenDialog(HWND hwndOwner, LPSTR filter, LPSTR fil, UINT iFilLen,
                LPSTR dlgtitle, LPSTR filtitle, UINT iFilTitleLen);
VOID EnableOpenGL( HWND hWnd, HDC * hDC, HGLRC * hRC );
VOID DisableOpenGL( HWND hWnd, HDC hDC, HGLRC hRC );
VOID DrawDragon();
VOID SaveDragon(char* filename, HWND hWnd);
VOID LoadDragon(char* filename, HWND hWnd);

// window vars
HDC hDC;
HGLRC hRC;
HDC hInsideDC;
HGLRC hInsideRC;
HWND hToolbar;
HWND hInsideWnd;

// drawing vars
bool solid = true;
bool convexHull = false;
bool nextList = false;
bool viewChanged = false;

// color vars
float approxColor[4] = {0.0f, 0.0f, 0.0f, 0.05f};
float approxColorNext[4] = {0.0f, 1.0f, 0.0f, 0.05f};
float insideColor[4] = {0.0f, 0.0f, 1.0f, 0.05f};
float insideColorNext[4] = {1.0f, 0.0f, 0.0f, 0.05f};
float boundaryColor[4] = {0.0f, 0.0f, 0.0f, 0.05f};

Dragon* dragon = new Dragon(-0.5, 0.0, 0.5, 0.0);
Dragon* dragonNext = new Dragon(0.5, 0.0, 1.5, 0.0);

// WinMain
int WINAPI WinMain( HINSTANCE hInstance,
					HINSTANCE hPrevInstance,
					LPSTR lpCmdLine,
					int iCmdShow )
{
	WNDCLASS wcMain, wcInside;
	HWND hMainWnd;		// Dragon approx. window
	MSG msg;
	BOOL bQuit = FALSE;

	// register inside window class
	wcInside.style = CS_OWNDC;
	wcInside.lpfnWndProc = WndProc;
	wcInside.cbClsExtra = 0;
	wcInside.cbWndExtra = 0;
	wcInside.hInstance = hInstance;
	wcInside.hIcon = LoadIcon( NULL, IDI_APPLICATION );
	wcInside.hCursor = LoadCursor( NULL, IDC_ARROW );
	wcInside.hbrBackground = (HBRUSH)GetStockObject( WHITE_BRUSH );
	wcInside.lpszMenuName  = NULL;
	wcInside.lpszClassName = "Inside Dragon";
	RegisterClass( &wcInside );

	// create inside window
	hInsideWnd = CreateWindow("Inside Dragon", "Inside Dragon", 
							  WS_CAPTION | WS_VISIBLE,
							  501, 0, 500, 525,
							  NULL, NULL, hInstance, NULL );

	// enable OpenGL window
	EnableOpenGL( hInsideWnd, &hInsideDC, &hInsideRC );
	
	// register main window class
	wcMain.style = CS_OWNDC;
	wcMain.lpfnWndProc = WndProc;
	wcMain.cbClsExtra = 0;
	wcMain.cbWndExtra = 0;
	wcMain.hInstance = hInstance;
	wcMain.hIcon = LoadIcon( NULL, IDI_APPLICATION );
	wcMain.hCursor = LoadCursor( NULL, IDC_ARROW );
	wcMain.hbrBackground = (HBRUSH)GetStockObject( WHITE_BRUSH );
	wcMain.lpszMenuName  = NULL;
	wcMain.lpszClassName = "Dragon";
	RegisterClass( &wcMain );

	// create approximation window
	hMainWnd = CreateWindow("Dragon", "Dragon", 
							WS_CAPTION | WS_POPUPWINDOW | WS_VISIBLE | WS_MINIMIZEBOX,
							0, 0, 500, 525,
							NULL, NULL, hInstance, NULL );

	// enable OpenGL window
	EnableOpenGL( hMainWnd, &hDC, &hRC );

	// create and show toolbar
    hToolbar = CreateDialog( hInstance, 
							 MAKEINTRESOURCE(IDD_TOOLBAR), 
							 NULL, 
							 ToolbarDlgProc );
	ShowWindow( hToolbar, iCmdShow );

	DrawDragon();
	
	// program main loop
	while(GetMessage(&msg,0,0,0) == TRUE) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	// shutdown OpenGL
	DisableOpenGL( hMainWnd, hDC, hRC );
	DisableOpenGL( hInsideWnd, hInsideDC, hInsideRC );

	// destroy the window explicitly
	DestroyWindow( hMainWnd );

	return msg.wParam;
}


// Window Procedure

LRESULT CALLBACK WndProc( HWND hWnd, UINT message,
						  WPARAM wParam, LPARAM lParam )
{
	static float xCoordOld, yCoordOld;
	static float xChange, yChange;
	static float xSize, ySize;
	static bool lmouseDown;
	static char buffer[20];

	switch ( message )
	{

		case WM_CREATE:
			return TRUE;

		case WM_CLOSE:
			PostQuitMessage( 0 );
			return TRUE;

		case WM_PAINT:
			DrawDragon();
			return TRUE;

		case WM_MOUSEMOVE:
			if (lmouseDown)
			{
				xChange = LOWORD( lParam ) - xCoordOld;
				yChange = HIWORD( lParam ) - yCoordOld;
				xSize = (Camera::right - Camera::left);
				ySize = (Camera::top - Camera::bottom);

				Camera::right -= (xChange / 500) * xSize;
				Camera::left -= (xChange / 500) * xSize;
				Camera::top += (yChange / 500) * ySize;
				Camera::bottom += (yChange / 500) * ySize;

				xCoordOld = LOWORD( lParam );
				yCoordOld = HIWORD( lParam );

				// set camera settings in dialog
				gcvt( Camera::top, 4, buffer );
				SetWindowText( GetDlgItem( hToolbar, IDC_CTOP ), buffer );
				gcvt( Camera::bottom, 4, buffer );
				SetWindowText( GetDlgItem( hToolbar, IDC_CBOTTOM ), buffer );
				gcvt( Camera::left, 4, buffer );
				SetWindowText( GetDlgItem( hToolbar, IDC_CLEFT ), buffer );
				gcvt( Camera::right, 4, buffer );
				SetWindowText( GetDlgItem( hToolbar, IDC_CRIGHT ), buffer );
			}
			return TRUE;

		case WM_LBUTTONDOWN:
			xCoordOld = LOWORD( lParam );
			yCoordOld = HIWORD( lParam );
			lmouseDown = true;
			viewChanged = true;
			return TRUE;

		case WM_LBUTTONUP:
			lmouseDown = false;
			return TRUE;

		case WM_DESTROY:
			return TRUE;

		default:
			return DefWindowProc( hWnd, message, wParam, lParam );
	}
}

BOOL CALLBACK ToolbarDlgProc( HWND hDlg, UINT msg, 
							  WPARAM wParam, LPARAM lParam )
{
	// amount to pan camera by
	float delta = (float) ((Camera::right - Camera::left) / 50.0);
	static char buffer[20];
	static int x;
	static char filename[MAX_PATH];
	static char filetitle[MAX_PATH];
	static char filter[] = "Dragon Files (*.dra)\0*.dra\0All files (*.*)\0*.*";
	static Dragon* currentDragon;
	static int level;
	static int approx;
	
	switch( msg )
    {
		case WM_COMMAND:
            
			switch( LOWORD(wParam) )
            {
				// zoom in
				case IDC_ZOOMIN:
					EnableWindow(GetDlgItem(hDlg, IDC_ZOOMIN), FALSE);
					SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Zooming" );
					for (x = 0; x < 10; x++)
					{
						Camera::top -= delta;
						Camera::bottom += delta;
						Camera::left += delta;
						Camera::right -= delta;
						viewChanged = true;
						DrawDragon();
					}
					SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Idle" );
					EnableWindow(GetDlgItem(hDlg, IDC_ZOOMIN), TRUE);
					break;

				// zoom out
				case IDC_ZOOMOUT:
					EnableWindow(GetDlgItem(hDlg, IDC_ZOOMOUT), FALSE);
					SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Zooming" );
					for (x = 0; x < 10; x++)
					{
						Camera::top += delta;
						Camera::bottom -= delta;
						Camera::left -= delta;
						Camera::right += delta;
						DrawDragon();
					}
					SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Idle" );
					EnableWindow(GetDlgItem(hDlg, IDC_ZOOMOUT), TRUE);
					break;

				// split visibles
				case IDC_SPLIT:
					EnableWindow(GetDlgItem(hDlg, IDC_SPLIT), FALSE);
					if (nextList)
					{
						currentDragon = dragonNext;
						level = IDC_LEVEL2;
						approx = IDC_APPROX2;
					}
					else
					{
						currentDragon = dragon;
						level = IDC_LEVEL;
						approx = IDC_APPROX;
					}
					if (viewChanged)
					{
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Culling polygons" );
						currentDragon->CompactVisibleApprox(Camera::top, Camera::bottom, Camera::left, Camera::right);
					}
					SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Splitting polygons" );
					if (currentDragon->Split(hToolbar, IDC_STATUS))
					{
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Calculating inside" );
						currentDragon->CompactVisibleInside(Camera::top, Camera::bottom, Camera::left, Camera::right);
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Idle" );
						itoa( currentDragon->GetSplits(), buffer, 10 );
						SetWindowText( GetDlgItem( hToolbar, level ), buffer );
						itoa( currentDragon->GetApproxTotal(), buffer, 10 );
						SetWindowText( GetDlgItem( hToolbar, approx ), buffer );
					}
					else
					{
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Please zoom more" );						
					}
					viewChanged = false;
					EnableWindow(GetDlgItem(hDlg, IDC_SPLIT), TRUE);
					break;

				case IDC_NEXT:
					nextList = !nextList;
					viewChanged = true;
					break;

				case IDC_SAVE:
					strcpy(filename,"");
  					if (OpenDialog(hDlg, filter, filename, MAX_PATH, 
								   "Save file", filetitle, MAX_PATH))
					{
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Saving" );
  						SaveDragon(filename, hDlg);
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Idle" );
					}
					break;

				case IDC_LOAD:
					strcpy(filename,"");
  					if (OpenDialog(hDlg, filter, filename, MAX_PATH, 
								   "Load file", filetitle, MAX_PATH))
					{
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Loading" );
						LoadDragon(filename, hDlg);
						SetWindowText( GetDlgItem( hToolbar, IDC_STATUS ), "Idle" );
						itoa( dragon->GetSplits(), buffer, 10 );
						SetWindowText( GetDlgItem( hToolbar, IDC_LEVEL ), buffer );
						itoa( dragonNext->GetSplits(), buffer, 10 );
						SetWindowText( GetDlgItem( hToolbar, IDC_LEVEL2 ), buffer );
					}
					break;

			}
			DrawDragon();
		break;

		default:
            return FALSE;
 
	}

    return TRUE;
}

VOID SaveDragon(char* filename, HWND hWnd)
{
	ofstream fout;
	int splits;
	double hundred = -100;

	fout.open(filename);
	fout.write((char*)&Camera::top, sizeof(float));
	fout.write((char*)&Camera::bottom, sizeof(float));
	fout.write((char*)&Camera::left, sizeof(float));
	fout.write((char*)&Camera::right, sizeof(float));
	splits = dragon->GetSplits();
	fout.write((char*)&splits, sizeof(int));
	dragon->Dump(fout);
	fout.write((char*)&hundred, sizeof(double));
	splits = dragonNext->GetSplits();
	fout.write((char*)&splits, sizeof(int));
	dragonNext->Dump(fout);
	fout.close();
}

VOID LoadDragon(char* filename, HWND hWnd)
{
	ifstream fin;
	float dummyF;
	float backup[4];
	static char buffer[20];

	fin.open(filename);
	backup[0] = Camera::top;
	backup[1] = Camera::bottom;
	backup[2] = Camera::left;
	backup[3] = Camera::right;

	if (!fin.eof())
	{
		fin.read((char *)&dummyF, sizeof(float));
		Camera::top = dummyF;
	}
	else
	{
		Camera::top = backup[0];
		Camera::bottom = backup[1];
		Camera::left = backup[2];
		Camera::right = backup[3];
		return;
	}
	if (!fin.eof())
	{
		fin.read((char *)&dummyF, sizeof(float));
		Camera::bottom = dummyF;
	}
	else
	{
		Camera::top = backup[0];
		Camera::bottom = backup[1];
		Camera::left = backup[2];
		Camera::right = backup[3];
		return;
	}
	if (!fin.eof())
	{
		fin.read((char *)&dummyF, sizeof(float));
		Camera::left = dummyF;
	}
	else
	{
		Camera::top = backup[0];
		Camera::bottom = backup[1];
		Camera::left = backup[2];
		Camera::right = backup[3];
		return;
	}
	if (!fin.eof())
	{
		fin.read((char *)&dummyF, sizeof(float));
		Camera::right = dummyF;
	}
	else
	{
		Camera::top = backup[0];
		Camera::bottom = backup[1];
		Camera::left = backup[2];
		Camera::right = backup[3];
		return;
	}

	dragon->Load(fin);
	dragonNext->Load(fin);
}

// Dragon drawing routine
VOID DrawDragon()
{
	//----- draw the approximation window
	wglMakeCurrent( hDC, hRC );

	// load view frustum settings
	glMatrixMode( GL_PROJECTION );
		glLoadIdentity();
		glOrtho( Camera::left,
				 Camera::right,
				 Camera::bottom,
				 Camera::top,
				 -1.0, 
				 2.0 );

	// switch back to draw mode	
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

	// clear buffer
	glClearColor( 1.0f, 1.0f, 1.0f, 0.0f );
	glClear( GL_COLOR_BUFFER_BIT );

	// translate and draw dragon
	if (nextList)
	{
		dragon->DrawApprox( approxColor );
		dragonNext->DrawApprox( approxColorNext );
	}
	else
	{
		dragon->DrawApprox( approxColorNext );
		dragonNext->DrawApprox( approxColor );
	}

	// swap buffer to screen
	SwapBuffers( hDC );
	
	// ----- draw the inside window
	wglMakeCurrent( hInsideDC, hInsideRC );

	// load view frustum settings
	glMatrixMode( GL_PROJECTION );
		glLoadIdentity();
		glOrtho( Camera::left,
				 Camera::right,
				 Camera::bottom,
				 Camera::top,
				 -10.0, 
				 1.0 );

	// switch back to draw mode	
	glMatrixMode( GL_MODELVIEW );
		glLoadIdentity();

	// clear buffer
	glClearColor( 1.0f, 1.0f, 1.0f, 0.0f );
	glClear( GL_COLOR_BUFFER_BIT );

	// translate and draw dragon
	if (nextList)
	{
		dragon->DrawInside( insideColor, boundaryColor );
		dragonNext->DrawInside( insideColorNext, boundaryColor );
	}
	else
	{
		dragon->DrawInside( insideColorNext, boundaryColor );
		dragonNext->DrawInside( insideColor, boundaryColor );
	}

	// swap buffer to screen
	SwapBuffers( hInsideDC );

	return;
}

BOOL OpenDialog(HWND hwndOwner, LPSTR filter, LPSTR fil, UINT iFilLen,
                LPSTR dlgtitle=NULL, LPSTR filtitle=NULL, UINT iFilTitleLen=0)
{
    OPENFILENAME opfil;
    memset((LPOPENFILENAME)&opfil,0,sizeof(opfil));
    opfil.lStructSize = sizeof(opfil);
    opfil.hInstance = (HINSTANCE)GetWindowLong(hwndOwner,GWL_HINSTANCE);
                                                //application instance
    opfil.lpstrFilter = filter;                 //filter of files separated by \0
    opfil.lpstrFile = fil;                      //absolute path of filename
    opfil.nMaxFile = iFilLen;                   //length of filename buffer    
    opfil.lpstrFileTitle = filtitle;            //filename with no path
    opfil.nMaxFileTitle = iFilTitleLen;         //length of filename buffer
    opfil.lpstrTitle = dlgtitle;                //title of dialog box
    opfil.Flags = OFN_HIDEREADONLY;             //optional flags    
    return GetOpenFileName(&opfil);    
}

// Enable OpenGL
VOID EnableOpenGL( HWND hWnd, HDC * hDC, HGLRC * hRC )
{
	PIXELFORMATDESCRIPTOR pfd;
	int iFormat;

	// get the device context (DC)
	*hDC = GetDC( hWnd );

	// set the pixel format for the DC
	ZeroMemory( &pfd, sizeof( pfd ) );
	pfd.nSize = sizeof( pfd );
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | 
				  PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 8;
	pfd.cDepthBits = 8;
	pfd.iLayerType = PFD_MAIN_PLANE;
	iFormat = ChoosePixelFormat( *hDC, &pfd );
	SetPixelFormat( *hDC, iFormat, &pfd );

	// create and enable the render context (RC)
	*hRC = wglCreateContext( *hDC );
	wglMakeCurrent( *hDC, *hRC );
	glEnableClientState(GL_VERTEX_ARRAY);
}


// Disable OpenGL
VOID DisableOpenGL( HWND hWnd, HDC hDC, HGLRC hRC )
{
	glDisableClientState(GL_VERTEX_ARRAY);
	wglMakeCurrent( NULL, NULL );
	wglDeleteContext( hRC );
	ReleaseDC( hWnd, hDC );
}
