///////////////////////////////////////////////////////////////////////////////////
// File : APSF.h
///////////////////////////////////////////////////////////////////////////////////
//
// Copyright 2006
// The University of North Carolina at Chapel Hill
// 
///////////////////////////////////////////////////////////////////////////////////
//
// Permission to use, copy, modify, distribute and sell this software and its 
// documentation for any purpose is hereby granted without fee, provided that 
// the above copyright notice appear in all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// Binaries may be compiled with this software without any royalties or 
// restrictions. 
//
// The University of North Carolina at Chapel Hill makes no representations 
// about the suitability of this software for any purpose. It is provided 
// "as is" without express or implied warranty.
//

#ifndef APSF_H
#define APSF_H

#include <cmath>
#include <gl/glut.h>
#include <vector>
#include "ppm\ppm.hpp"

using namespace std;

#ifndef M_PI
#define M_PI 3.1415926535897931f
#endif

////////////////////////////////////////////////////////////////////
/// \brief Generates the rendering filter
////////////////////////////////////////////////////////////////////
class APSF  
{
public:
  //! constructor
	APSF(int res = 512);
  //! destructor
	virtual ~APSF();

  //! read in an APSF file
  void read(const char* filename);
  //! write out an APSF file
  void write(const char* filename);
  //! write out the current kernel to a PPM file
  void writePPM(const char* filename);
  
  //! generate one line of the kernel and spin it radially
  void generateKernelFast();

  //! resolution of current kernel
  int res() { return _res; };
  
  //! returns the float array for the kernel
  float* kernel() { return _kernel; };
  
private:
  //! kernel resolution
  int     _res;
  //! convolution kernel
  float*  _kernel;

  ////////////////////////////////////////////////////////////////////
  // APSF components
  ////////////////////////////////////////////////////////////////////

  // scattering parameters
  float _q;
  float _T;
  float _I0;
  float _sigma;
  float _R;
  float _D;

  float _retinaSize;
  float _eyeSize;
  
  //! number of coefficients
  int _maxTerms;
  
  //! function value at a point
  float pointAPSF(float mu);
  
  ////////////////////////////////////////////////////////////////////
  // auxiliary functions
  ////////////////////////////////////////////////////////////////////
  float legendreM(int m, float mu);
  float gM(float I0, int m) {
    return (m == 0) ? 0.0f : exp(-(betaM(m, _q) * _T + alphaM(m) * log(_T)));
  };
  float alphaM(float m) { 
    return m + 1.0f;
  };
  float betaM(float m, float q) {
    return ((2.0f * m + 1.0f) / m) * (1.0f - pow(q, (int)m - 1));
  };
  float factorial(float x) {
    return (x <= 1.0f) ? 1.0f : x * factorial(x - 1.0f);
  };
  float choose(float x, float y) {
    return factorial(x) / (factorial(y) * factorial(x - y));
  };
};

#endif
