///////////////////////////////////////////////////////////////////////////////////
// File : CELL.h
///////////////////////////////////////////////////////////////////////////////////
//
// Copyright 2006
// The University of North Carolina at Chapel Hill
// 
///////////////////////////////////////////////////////////////////////////////////
//
// Permission to use, copy, modify, distribute and sell this software and its 
// documentation for any purpose is hereby granted without fee, provided that 
// the above copyright notice appear in all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// Binaries may be compiled with this software without any royalties or 
// restrictions. 
//
// The University of North Carolina at Chapel Hill makes no representations 
// about the suitability of this software for any purpose. It is provided 
// "as is" without express or implied warranty.
//

#ifndef CELL_H
#define CELL_H
#include <cstdlib>

//////////////////////////////////////////////////////////////////////
/// \enum Possible states of the cell in the DBM simulation
//////////////////////////////////////////////////////////////////////
enum CELL_STATE {EMPTY, NEGATIVE, POSITIVE, REPULSOR, ATTRACTOR};

//////////////////////////////////////////////////////////////////////
/// \brief Basic cell data structure of the quadtree
//////////////////////////////////////////////////////////////////////
class CELL  
{
public:
  //! normal cell constructor  
  CELL(float north, 
       float east, 
       float south, 
       float west, 
       CELL* parent = NULL, 
       int depth = 0);
  
  //! ghost cell constructor  
  CELL(int depth = 0);

  //! destructor
  ~CELL();
  
  //! The children of the node in the quadtree
  /*! 
      Winding order of children is:

      \verbatim
        _________
        |   |   |
        | 0 | 1 |  
        |___|___|
        |   |   |
        | 3 | 2 |   
        |___|___|
      \endverbatim */
  CELL* children[4]; 
  
  //! The physical bounds of the current grid cell
  /*! 
      Winding order of bounds is:

      \verbatim
        0 - north
        1 - east
        2 - south
        3 - west
      \endverbatim */
  float bounds[4];

  //! The neighbors in the balanced quadtree
  /*!
    winding order of the neighbors is:
  
    \verbatim
         | 0  | 1  |
     ____|____|____|_____
         |         |
       7 |         |  2
     ____|         |_____
         |         |
       6 |         |  3
     ____|_________|_____
         |    |    |
         | 5  |  4 |
    \endverbatim
 
    Neighbors 0,2,4,6 should always exist. Depending on
    if the neighbor is on a lower refinement level,
    neighbors 1,3,5,7 may or may not exist. If they are not
    present, the pointer value should ne NULL.  */
  CELL* neighbors[8];

  //! Poisson stencil coefficients
  /*!
    winding order of the stencil coefficients:
  
    \verbatim
         | 0  | 1  |
     ____|____|____|_____
         |         |
      7  |         |  2
     ____|    8    |_____
         |         |
      6  |         |  3
     ____|_________|_____
         |    |    |
         | 5  | 4  |
    \endverbatim
    Stencils 0,2,4,6 should always exist. Depending on
    if the neighbor is on a lower refinement level,
    stencils 1,3,5,7 may or may not exist. If they are not
    present, the pointer value should ne NULL.    */
  float stencil[9];
  
  float center[2];    ///< center of the cell
  int depth;          ///< current tree depth
  bool candidate;     ///< already a member of candidate list?

  CELL* parent;       ///< parent node in the quadtree
  CELL_STATE state;   ///< DBM state of the cell

  void refine();      ///< subdivide the cell

  ////////////////////////////////////////////////////////////////
  // solver-related variables
  ////////////////////////////////////////////////////////////////
  bool boundary;      ///< boundary node to include in the solver?
  float potential;    ///< current electric potential
  float b;            ///< rhs of the linear system
  float residual;     ///< residual in the linear solver
  int index;          ///< lexicographic index for the solver

  ////////////////////////////////////////////////////////////////
  // neighbor lookups
  ////////////////////////////////////////////////////////////////
  CELL* northNeighbor();  ///< lookup northern neighbor
  CELL* southNeighbor();  ///< lookup southern neighbor
  CELL* westNeighbor();   ///< lookup western neighbor
  CELL* eastNeighbor();   ///< lookup eastern neighbor
};

#endif
