///////////////////////////////////////////////////////////////////////////////////
// File : DAG.h
///////////////////////////////////////////////////////////////////////////////////
//
// Copyright 2006
// The University of North Carolina at Chapel Hill
// 
///////////////////////////////////////////////////////////////////////////////////
//
// Permission to use, copy, modify, distribute and sell this software and its 
// documentation for any purpose is hereby granted without fee, provided that 
// the above copyright notice appear in all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// Binaries may be compiled with this software without any royalties or 
// restrictions. 
//
// The University of North Carolina at Chapel Hill makes no representations 
// about the suitability of this software for any purpose. It is provided 
// "as is" without express or implied warranty.
//

#ifndef DAG_H
#define DAG_H

#include <map>
#include <vector>
#include <cmath>
#include <gl/glut.h>
#include <iostream>

using namespace std;

////////////////////////////////////////////////////////////////////
/// \brief Directed acyclic graph that renders the final lightning
////////////////////////////////////////////////////////////////////
class DAG  
{
public:
  //! constructor
	DAG(int xRes, int yRes);
  //! destructor
	virtual ~DAG();

  //! build the stepped ladder
  void buildLeader(int bottomHit);

  //! add DAG segment
  bool addSegment(int index, int neighbor);

  //! draw to OpenGL
  void draw() { drawNode(_root); };

  //! draw to an offscreen buffer
  float*& drawOffscreen(int scale = 1);

  //! read in a new DAG
  void read(const char* filename);
  //! write out the current DAG
  void write(const char* filename);

  //! quadtree x resolution accessor
  int xRes() { return _xRes; };
  //! quadtree y resolution accessor
  int yRes() { return _yRes; };
 
  //! input image x resolution accessor
  int& inputWidth() { return _inputWidth; };
  //! input image y resolution accessor
  int& inputHeight() { return _inputHeight; };
  
private:
  //! x resolution of quadtree
  int _xRes;
  //! y resolution of quadtree
  int _yRes;
  //! physical length of one grid cell
  float _dx;
  //! physical length of one grid cell
  float _dy;
  
  ////////////////////////////////////////////////////////////////////
  /// \brief node for line segment tree
  ////////////////////////////////////////////////////////////////////
  struct NODE {
    int index;
    vector<NODE*> neighbors;
    NODE* parent;
    bool leader;
    bool secondary;
    int depth;
    NODE* maxDepthNode;
    float intensity;

    NODE(int indexIn) { 
      index = indexIn; 
      parent = NULL;
      leader = false;
      depth = 0;
      maxDepthNode = NULL;
    };
  };
  //! recursive destructor
  void deleteNode(NODE* root);

  //! root of DAG
  NODE* _root;

  //! hash table of DAG nodes
  map<int, NODE*> _hash;

  //! build side branch
  void buildBranch(NODE* node, int depth);
  //! draw a node to OpenGL
  void drawNode(NODE* root);
  //! find the deepest node in a given subtree
  void findDeepest(NODE* root, NODE*& deepest);

  //! read in a DAG node 
  void readNode(FILE* file);
  //! write out a DAG node 
  void writeNode(NODE* root, FILE* file);

  //! total number of nodes in scene
  int _totalNodes;

  //! node that finally hit bottom
  int _bottomHit;

  //! set the line segment intensities
  void buildIntensity(NODE* root);
  
  //! brightness of secondary branch
  float _secondaryIntensity;
  //! brightness of primary branch
  float _leaderIntensity;

  ////////////////////////////////////////////////////////////////
  // offscreen buffer variables
  ////////////////////////////////////////////////////////////////
  //! offscreen buffer
  float* _offscreenBuffer;
  
  //! width of offscreen buffer
  int _width;
  
  //! height of offscreen buffer
  int _height;
  
  //! scale of offscreen buffer compared to original image
  int _scale;
  
  //! draw a given node in the DAG
  void drawOffscreenNode(NODE* root);

  //! rasterize a single line to the offscreen buffer
  void drawLine(int begin[], int end[], float intensity);

  //! input image x resolution
  int _inputWidth;
  //! input image y resolution
  int _inputHeight;
};

#endif
